/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpPKEMod.c,v 1.7 2001/01/25 22:11:16 jeffc Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"

#include <stdio.h>
#include <string.h>

#include "pgpDebug.h"
#include "pgpMakePKE.h"
#include "pgpPKEMod.h"
#include "pgpPktByte.h"
#include "pgpFIFO.h"
#include "pgpJoin.h"
#include "pgpMem.h"
#include "pgpErrors.h"
#include "pgpPipeline.h"
#include "pgpContext.h"

#define PKEMODMAGIC	0x90b11c47

typedef struct PKEModContext {
	PGPPipeline		pipe;
	
	PGPByte *key;
	size_t keylen;
	PGPByte *buf;
	size_t buflen;
	PgpVersion version;
	PGPPipeline *tail;
	DEBUG_STRUCT_CONSTRUCTOR( PKEModContext )
} PKEModContext;

static PGPError
Flush (PGPPipeline *myself)
{
	PKEModContext *context;

	pgpAssert (myself);
	pgpAssert (myself->magic == PKEMODMAGIC);

	context = (PKEModContext *)myself->priv;
	pgpAssert (context);
	pgpAssert (context->tail);

	return context->tail->flush (context->tail);
}

static size_t
Write (PGPPipeline *myself, PGPByte const *buf, size_t size, PGPError *error)
{
	PKEModContext *context;
	PGPContextRef	cdkContext;
	
	pgpAssertAddrValid( myself, PGPPipeline );
	cdkContext	= myself->cdkContext;

	pgpAssert (myself);
	pgpAssert (myself->magic == PKEMODMAGIC);
	pgpAssert (error);

	context = (PKEModContext *)myself->priv;
	pgpAssert (context);
	pgpAssert (context->tail);

	/* Once data is written, clear out the key! */
	if (context->key) {
		pgpClearMemory (context->key, context->keylen);
		pgpContextMemFree( cdkContext, context->key);
		context->key = NULL;
		if (context->buf) {
			pgpClearMemory (context->buf, context->buflen);
			pgpContextMemFree( cdkContext, context->buf);
			context->buf = NULL;
			context->buflen = 0;
		}
	}

	return context->tail->write (context->tail, buf, size, error);
}

static PGPError
Annotate (PGPPipeline *myself, PGPPipeline *origin, int type,
	  PGPByte const *string, size_t size)
{
	PKEModContext *context;

	pgpAssert (myself);
	pgpAssert (myself->magic == PKEMODMAGIC);

	context = (PKEModContext *)myself->priv;
	pgpAssert (context);
	pgpAssert (context->tail);

	return context->tail->annotate (context->tail, origin, type,
					string, size);
}

static PGPError
SizeAdvise (PGPPipeline *myself, PGPFileOffset bytes)
{
	PKEModContext *context;

	pgpAssert (myself);
	pgpAssert (myself->magic == PKEMODMAGIC);

	context = (PKEModContext *)myself->priv;
	pgpAssert (context);
	pgpAssert (context->tail);

	return context->tail->sizeAdvise (context->tail, bytes);
}

static PGPError
Teardown (PGPPipeline *myself)
{
	PKEModContext *context;
	PGPContextRef	cdkContext;
	
	pgpAssertAddrValid( myself, PGPPipeline );
	cdkContext	= myself->cdkContext;

	pgpAssert (myself);
	pgpAssert (myself->magic == PKEMODMAGIC);

	context = (PKEModContext *)myself->priv;
	pgpAssert (context);

	if (context->tail)
		context->tail->teardown (context->tail);

	if (context->key) {
		pgpClearMemory (context->key, context->keylen);
		pgpContextMemFree( cdkContext, context->key);
		context->key = NULL;
	}
	if (context->buf) {
		pgpClearMemory (context->buf, context->buflen);
		pgpContextMemFree( cdkContext, context->buf);
		context->buf = NULL;
		context->buflen = 0;
	}
	pgpClearMemory( context,  sizeof (*context));
	pgpContextMemFree( cdkContext, context);
	
	return kPGPError_NoErr;
}

/*
 * This adds a new PKE packet to the head of the message.  If
 * context->key is NULL, return an error.  Also, return an error if
 * what we buffer is not as much as we thought we'd written
 */
	PGPError
pgpPkeAddKey (PGPPipeline *myself, PGPKeyDBObj const *pubkey)
{
	PKEModContext *	context;
	PGPSize			size;
	PGPError		err	= kPGPError_NoErr;
	PGPByte *		p	= NULL;
	PGPByte *		pkebuf = NULL;
	PGPSize			pkebuflen;
	PGPContextRef	cdkContext;
	
	pgpAssertAddrValid( myself, PGPPipeline );
	cdkContext	= myself->cdkContext;

	pgpAssert (myself);
	pgpAssert (myself->magic == PKEMODMAGIC);

	context = (PKEModContext *)myself->priv;
	pgpAssert (context);
	pgpAssert (context->tail);

	err = makePke( cdkContext, pubkey, context->key, context->keylen,
				   context->version, &pkebuf, &pkebuflen );
	if( IsPGPError( err ) )
		return err;

	size = pkebuflen + 3;
	if (context->buflen < size) {
		if( IsNull( context->buf ) ) {
			context->buf = (PGPByte *)pgpContextMemAlloc( cdkContext, size,
											   0 );
			if( IsNull( context->buf ) )
				err = kPGPError_OutOfMemory;
		} else {
			err = pgpContextMemRealloc( cdkContext,
					(void **)&context->buf, size, 0 );
		}
		if( IsPGPError( err ) ) {
			PGPFreeData( pkebuf );
			return err;
		}
		context->buflen = size;
	}

	pgpCopyMemory( pkebuf, context->buf+3, pkebuflen );
	PGPFreeData( pkebuf );
	pkebuf = NULL;

	if (context->version > PGPVERSION_3) {
		if (PKTLEN_ONE_BYTE(pkebuflen)) {
			context->buf[1] = PKTBYTE_BUILD_NEW(PKTBYTE_ESK);
			context->buf[2] = PKTLEN_1BYTE(pkebuflen);
			p = context->buf+1;
			size = pkebuflen + 2;
		} else {
			context->buf[0] = PKTBYTE_BUILD_NEW(PKTBYTE_ESK);
			context->buf[1] = PKTLEN_BYTE0(pkebuflen);
			context->buf[2] = PKTLEN_BYTE1(pkebuflen);
			p = context->buf;
			size = pkebuflen + 3;
		}
	} else {
#if 0
/* PGP 2.X wants to see 2 byte length field */
		if (pkebuflen < 256) {
			context->buf[1] = PKTBYTE_BUILD (PKTBYTE_ESK, 0);
			context->buf[2] = (PGPByte)pkebuflen;
			p = context->buf+1;
			size = pkebuflen + 2;
		} else {}
#endif
		context->buf[0] = PKTBYTE_BUILD (PKTBYTE_ESK, 1);
		context->buf[1] = (PGPByte)(pkebuflen >> 8);
		context->buf[2] = (PGPByte)pkebuflen;
		p = context->buf;
		size = pkebuflen + 3;
	}

	pgpAssert( IsntNull( p ) );
	if (pgpJoinBuffer (context->tail, p, size) != size)
		return kPGPError_OutOfMemory;

	return kPGPError_NoErr;
}

PGPPipeline **
pgpPkeCreate (
	PGPContextRef	cdkContext,
	PGPPipeline **	head,
	PgpVersion		version,
	PGPByte const *	key,
	size_t			keylen)
{
	PGPPipeline *mod, *joinhead = NULL, **tail;
	PKEModContext *context;
	PGPByte *buf;

	pgpAssert (key);

	if (!head)
		return NULL;

	context = (PKEModContext *)pgpContextMemAlloc( cdkContext,
		sizeof (*context), kPGPMemoryMgrFlags_Clear);
	if (!context)
		return NULL;
	mod = &context->pipe;
	
	tail = pgpJoinCreate ( cdkContext, &joinhead, &pgpByteFifoDesc);
	if (!tail) {
		pgpContextMemFree( cdkContext, context);
		return NULL;
	}
	buf = (PGPByte *)
		pgpContextMemAlloc( cdkContext, keylen, kPGPMemoryMgrFlags_Clear);
	if (!buf) {
		pgpContextMemFree( cdkContext, context);
		joinhead->teardown (joinhead);
		return NULL;
	}
	memcpy (buf, key, keylen);

	mod->magic = PKEMODMAGIC;
	mod->write = Write;
	mod->flush = Flush;
	mod->sizeAdvise = SizeAdvise;
	mod->annotate = Annotate;
	mod->teardown = Teardown;
	mod->name = "PKE Module";
	mod->priv = context;
	mod->cdkContext	= cdkContext;

	context->key = buf;
	context->keylen = keylen;
	context->version = version;

	context->tail = joinhead;
	*tail = *head;
	*head = mod;
	return tail;
}




/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
